import React, { useMemo, useRef, useEffect, useState } from 'react';
import {
  Table, TableBody, TableCell, TableContainer, TableHead, TableRow, Paper, Tooltip
} from '@mui/material';
import { useSelector, useDispatch } from 'react-redux';
import { devicesActions } from '../store';

import { DndContext, closestCenter } from '@dnd-kit/core';
import { arrayMove, SortableContext, useSortable, horizontalListSortingStrategy } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';

import BatteryFullIcon from '@mui/icons-material/BatteryFull';
import BatteryStdIcon from '@mui/icons-material/BatteryStd';
import BatteryAlertIcon from '@mui/icons-material/BatteryAlert';
import BatteryChargingFullIcon from '@mui/icons-material/BatteryChargingFull';
import PowerIcon from '@mui/icons-material/Power';
import BoltIcon from '@mui/icons-material/Bolt';

import SignalCellular0BarIcon from '@mui/icons-material/SignalCellular0Bar';
import SignalCellular1BarIcon from '@mui/icons-material/SignalCellular1Bar';
import SignalCellular2BarIcon from '@mui/icons-material/SignalCellular2Bar';
import SignalCellular3BarIcon from '@mui/icons-material/SignalCellular3Bar';
import SignalCellular4BarIcon from '@mui/icons-material/SignalCellular4Bar';


// Função para escolher o ícone certo
const getRssiIcon = (rssi) => {
  if (typeof rssi !== 'number') return <SignalCellular0BarIcon style={{ color: 'gray' }} />;

  if (rssi <= 5) return <SignalCellular0BarIcon style={{ color: 'red' }} />;
  if (rssi <= 15) return <SignalCellular1BarIcon style={{ color: 'orange' }} />;
  if (rssi <= 25) return <SignalCellular3BarIcon style={{ color: 'goldenrod' }} />;
  return <SignalCellular4BarIcon style={{ color: 'green' }} />;
};

const ONLINE_MIN = 10;

function getStatus(device, position) {
  if (['online', 'offline', 'unknown'].includes(device?.status)) return device.status;
  const ts = position?.deviceTime || position?.fixTime || device?.lastUpdate;
  if (!ts) return 'unknown';
  const diffMin = (Date.now() - new Date(ts).getTime()) / 60000;
  return diffMin <= ONLINE_MIN ? 'online' : 'offline';
}

const statusColor = (s) => (s === 'online' ? '#2e7d32' : s === 'offline' ? '#c62828' : '#6b6b6b');
const statusLabel = (s) => (s === 'online' ? 'Conectado' : s === 'offline' ? 'Desconectado' : 'Sem dados');
const toKmh = (s) => (typeof s === 'number' ? (s * 1.852).toFixed(1) : '0.0');

const voltsToPercent = (v) => {
  if (typeof v !== 'number') return null;
  const min = 3.60, max = 4.20;
  const pct = Math.round(((v - min) / (max - min)) * 100);
  return Math.max(0, Math.min(100, pct));
};

const DIR = ['N', 'NNE', 'NE', 'ENE', 'L', 'ESE', 'SE', 'SSE', 'S', 'SSO', 'SO', 'OSO', 'O', 'ONO', 'NO', 'NNO', 'N'];
const headingLabel = (deg) => {
  if (typeof deg !== 'number') return '-';
  const i = Math.round((deg % 360) / 22.5);
  return `${DIR[i]} (${Math.round(deg)}°)`;
};

const defaultColumns = [
  'name', 'address', 'gps', 'lat', 'lon', 'speed', 'battery', 'ignition', 'charge',
  'blocked', 'motion', 'sat', 'hours', 'rssi', 'course', 'alt'
];

const columnLabels = {
  name: 'Nome', address: 'Endereço', gps: 'GPS', lat: 'Latitude', lon: 'Longitude',
  speed: 'Velocidade', battery: 'Bateria',
  ignition: 'Ignição', charge: 'Carregando', blocked: 'Bloqueado',
  motion: 'Movimento', sat: 'Satélites', hours: 'Horas',
  rssi: 'RSSI', course: 'Direção', alt: 'Altitude',
};

function SortableHeader({ column }) {
  const { attributes, listeners, setNodeRef, transform, transition } = useSortable({ id: column });
  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    cursor: 'grab',
    userSelect: 'none'
  };
  return (
    <TableCell
      ref={setNodeRef}
      style={style}
      {...attributes}
      {...listeners}
      sx={{
        backgroundColor: '#007bff !important', // azul fixo, tema claro ou escuro
        color: '#fff',
        fontWeight: 'bold',
        fontSize: '0.75rem',
        padding: '4px',
        whiteSpace: 'nowrap',
        textAlign: 'left'
      }}
    >
      {columnLabels[column]}
    </TableCell>
  );
}

export default function DeviceListTable({ devices }) {
  const dispatch = useDispatch();
  const positions = useSelector((state) => state.session.positions);
  const selectedDeviceId = useSelector((state) => state.devices.selectedId);
  const rowRefs = useRef({});
  const [columns, setColumns] = useState(defaultColumns);

  const handleClick = (id) => dispatch(devicesActions.selectId(id));

  useEffect(() => {
    if (selectedDeviceId && rowRefs.current[selectedDeviceId]) {
      rowRefs.current[selectedDeviceId].scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  }, [selectedDeviceId]);

  const idsKey = useMemo(() => devices.map((d) => String(d.id)).join('|'), [devices]);
  const posVersion = useMemo(() => {
    let max = 0;
    for (const p of Object.values(positions)) {
      const t = p?.deviceTime || p?.fixTime || p?.serverTime;
      const ts = t ? new Date(t).getTime() : 0;
      if (ts > max) max = ts;
    }
    return String(max);
  }, [positions]);

  const dt = useMemo(() => new Intl.DateTimeFormat('pt-BR', { dateStyle: 'short', timeStyle: 'short' }), []);
  const cellStyle = { fontSize: '0.65rem', padding: '4px', whiteSpace: 'nowrap', color: '#000', textAlign: 'left' };
  const nameCell = { ...cellStyle, maxWidth: 280 };     // Nome mais largo
  const addrCell = { ...cellStyle, maxWidth: 360, whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis' };

  return (
    <TableContainer
      component={Paper}
      style={{ height: '100%', overflow: 'auto', backgroundColor: '#fff' }}
    >
      <DndContext
        collisionDetection={closestCenter}
        onDragEnd={({ active, over }) => {
          if (active.id !== over?.id) {
            const oldIndex = columns.indexOf(active.id);
            const newIndex = columns.indexOf(over.id);
            setColumns(arrayMove(columns, oldIndex, newIndex));
          }
        }}
      >
        <Table key={`${idsKey}-${posVersion}`} stickyHeader size="small" sx={{ tableLayout: 'fixed' }}>
          <TableHead>
            <SortableContext items={columns} strategy={horizontalListSortingStrategy}>
              <TableRow>
                {columns.map((col) => (
                  <SortableHeader key={col} column={col} />
                ))}
              </TableRow>
            </SortableContext>
          </TableHead>
          <TableBody>
            {devices.map((device) => {
              const position = positions[device.id];
              const attrs = position?.attributes || {};
              const st = getStatus(device, position);
              const pct = typeof attrs.batteryLevel === 'number' ? attrs.batteryLevel : voltsToPercent(attrs.battery);
              const hasBattery = typeof pct === 'number' && !Number.isNaN(pct);
              const sats = attrs.sat ?? attrs.satellites;

              return (
                <TableRow
                  ref={(el) => { rowRefs.current[device.id] = el; }}
                  hover
                  key={`dev-${device.id}`}
                  onClick={() => handleClick(device.id)}
                  selected={selectedDeviceId === device.id}
                  sx={{
                    cursor: 'pointer',
                    '&.Mui-selected': { backgroundColor: '#c8facc !important' }, // verde claro fixo
                    '&:hover': { backgroundColor: '#f1fff1' }
                  }}
                >
                  {columns.map((col) => {
                    switch (col) {
                      case 'name':
                        return (
                          <TableCell key={col} sx={nameCell}>
                            <Tooltip title={device.name}>
                              <div style={{ display: 'flex', flexDirection: 'column' }}>
                                <span style={{
                                  fontWeight: 600,
                                  whiteSpace: 'nowrap',
                                  overflow: 'hidden',
                                  textOverflow: 'ellipsis'
                                }}>
                                  {device.name}
                                </span>
                                <span style={{ fontSize: '0.65rem', color: statusColor(st) }}>
                                  {statusLabel(st)}
                                </span>
                              </div>
                            </Tooltip>
                          </TableCell>
                        );
                      case 'address':
                        return (
                          <TableCell key={col} sx={addrCell}>
                            <Tooltip title={position?.address || 'N/D'}>
                              <span>{position?.address || 'N/D'}</span>
                            </Tooltip>
                          </TableCell>
                        );
                      case 'gps':
                        return <TableCell key={col} sx={cellStyle}>{position?.deviceTime ? dt.format(new Date(position.deviceTime)) : 'Sem posição'}</TableCell>;
                      case 'lat':
                        return <TableCell key={col} sx={cellStyle}>{typeof position?.latitude === 'number' ? position.latitude.toFixed(6) : '-'}</TableCell>;
                      case 'lon':
                        return <TableCell key={col} sx={cellStyle}>{typeof position?.longitude === 'number' ? position.longitude.toFixed(6) : '-'}</TableCell>;
                      case 'speed':
                        return <TableCell key={col} sx={cellStyle}>{position ? `${toKmh(position.speed)} km/h` : '-'}</TableCell>;
                      case 'battery':
                        return (
                          <TableCell key={col} sx={cellStyle}>
                            {hasBattery ? (
                              <Tooltip title={`Bateria: ${pct}%`}>
                                <span style={{ display: 'flex', alignItems: 'center', gap: 4 }}>
                                  {attrs.charge
                                    ? <BatteryChargingFullIcon style={{ color: 'green', fontSize: 16 }} />
                                    : pct >= 80
                                      ? <BatteryFullIcon style={{ color: 'green', fontSize: 16 }} />
                                      : pct >= 30
                                        ? <BatteryStdIcon style={{ color: 'orange', fontSize: 16 }} />
                                        : <BatteryAlertIcon style={{ color: 'red', fontSize: 16 }} />
                                  }
                                  {`${pct}%`}
                                </span>
                              </Tooltip>
                            ) : '-'}
                          </TableCell>
                        );
                      case 'ignition':
                        return <TableCell key={col} sx={cellStyle}><PowerIcon fontSize="small" color={attrs.ignition ? 'success' : 'error'} /></TableCell>;
                      case 'charge':
                        return <TableCell key={col} sx={cellStyle}><BoltIcon fontSize="small" style={{ color: attrs.charge ? 'green' : 'red' }} /></TableCell>;
                      case 'blocked':
                        return <TableCell key={col} sx={cellStyle}>{attrs.blocked ? 'Sim' : 'Não'}</TableCell>;
                      case 'motion':
                        return <TableCell key={col} sx={cellStyle}>{attrs.motion ? 'Sim' : 'Não'}</TableCell>;
                      case 'sat':
                        return <TableCell key={col} sx={cellStyle}>{sats ?? '-'}</TableCell>;
                      case 'hours':
                        return <TableCell key={col} sx={cellStyle}>{typeof attrs.hours === 'number' ? `${Math.floor(attrs.hours / 3600)} h` : '-'}</TableCell>;
                      case 'rssi':
                        return (
                          <TableCell key={col} sx={cellStyle}>
                            {attrs.rssi !== undefined ? (
                              <Tooltip title={`Sinal: ${attrs.rssi}`}>
                                {getRssiIcon(attrs.rssi)}
                              </Tooltip>
                            ) : '-'}
                          </TableCell>
                        );

                      case 'course':
                        return <TableCell key={col} sx={cellStyle}>{typeof position?.course === 'number' ? headingLabel(position.course) : '-'}</TableCell>;
                      case 'alt':
                        return <TableCell key={col} sx={cellStyle}>{typeof position?.altitude === 'number' ? position.altitude.toFixed(1) : '-'}</TableCell>;
                      default:
                        return <TableCell key={col} sx={cellStyle}>-</TableCell>;
                    }
                  })}
                </TableRow>
              );
            })}
          </TableBody>
        </Table>
      </DndContext>
    </TableContainer>
  );
}
