import { useDispatch, useSelector } from 'react-redux';
import { makeStyles } from 'tss-react/mui';
import {
  Avatar, IconButton, ListItemAvatar, ListItemButton, ListItemText, Tooltip, Typography,
} from '@mui/material';
import BatteryFullIcon from '@mui/icons-material/BatteryFull';
import BatteryChargingFullIcon from '@mui/icons-material/BatteryChargingFull';
import Battery60Icon from '@mui/icons-material/Battery60';
import BatteryCharging60Icon from '@mui/icons-material/BatteryCharging60';
import Battery20Icon from '@mui/icons-material/Battery20';
import BatteryCharging20Icon from '@mui/icons-material/BatteryCharging20';
import { devicesActions } from '../store';
import { formatPercentage } from '../common/util/formatter';
import { useTranslation } from '../common/components/LocalizationProvider';
import { mapIconKey, mapIcons } from '../map/core/preloadImages';
import { useAdministrator } from '../common/util/permissions';
import { useAttributePreference } from '../common/util/preferences';

const useStyles = makeStyles()((theme) => ({
  icon: {
    width: '18px',
    height: '18px',
    filter: 'brightness(0) invert(1)',
  },
  success: {
    color: theme.palette.success.main,
  },
  warning: {
    color: theme.palette.warning.main,
  },
  error: {
    color: theme.palette.error.main,
  },
}))

const DeviceRow = ({ data, index, style }) => {
  const { classes } = useStyles();
  const dispatch = useDispatch();
  const t = useTranslation();
  const admin = useAdministrator();

  const item = data[index];
  const position = useSelector((state) => state.session.positions[item.id]);
  const devicePrimary = useAttributePreference('devicePrimary', 'name');

  return (
    <div style={style}>
      <ListItemButton
        key={item.id}
        onClick={() => dispatch(devicesActions.selectId(item.id))}
        disabled={!admin && item.disabled}
        sx={{
          alignItems: 'flex-start',
          py: 0.5,
          px: 1.5,
          minHeight: 36,
        }}
      >
        <ListItemAvatar sx={{ minWidth: 36 }}>
          <Avatar sx={{ width: 26, height: 26 }}>
            <img className={classes.icon} src={mapIcons[mapIconKey(item.category)]} alt="" />
          </Avatar>
        </ListItemAvatar>

        <ListItemText
          primary={
            <div className="device-row">
              <Typography variant="caption" noWrap><strong>{item[devicePrimary]}</strong></Typography>

              {position ? (
                <>
                  <Typography variant="caption" noWrap><strong>GPS:</strong> {new Date(position.deviceTime).toLocaleString()}</Typography>
                  <Typography variant="caption" noWrap><strong>Lat:</strong> {position.latitude.toFixed(6)}</Typography>
                  <Typography variant="caption" noWrap><strong>Lon:</strong> {position.longitude.toFixed(6)}</Typography>
                  <Typography variant="caption" noWrap><strong>Velocidade:</strong> {(position.speed * 1.852).toFixed(1)} km/h</Typography>
                  <Typography variant="caption" noWrap><strong>Endereço:</strong> {position.address || 'N/D'}</Typography>
                </>
              ) : (
                <Typography variant="caption" noWrap>Sem posição</Typography>
              )}
            </div>
          }
        />

        {/* Ícone da bateria */}
        {position?.attributes?.batteryLevel !== undefined && (
          <Tooltip title={`Bateria: ${formatPercentage(position.attributes.batteryLevel)}`}>
            <IconButton size="small">
              {(position.attributes.batteryLevel > 70 && (
                position.attributes.charge
                  ? (<BatteryChargingFullIcon fontSize="small" className={classes.success} />)
                  : (<BatteryFullIcon fontSize="small" className={classes.success} />)
              )) || (position.attributes.batteryLevel > 30 && (
                position.attributes.charge
                  ? (<BatteryCharging60Icon fontSize="small" className={classes.warning} />)
                  : (<Battery60Icon fontSize="small" className={classes.warning} />)
              )) || (
                position.attributes.charge
                  ? (<BatteryCharging20Icon fontSize="small" className={classes.error} />)
                  : (<Battery20Icon fontSize="small" className={classes.error} />)
              )}
            </IconButton>
          </Tooltip>
        )}
      </ListItemButton>
    </div>
  );
};

export default DeviceRow;
