import React, { useMemo, useState, useEffect } from 'react';
import {
  Table, TableBody, TableCell, TableContainer, TableHead, TableRow, Paper,
  Tooltip, IconButton, Checkbox, Menu, MenuItem, ListItemIcon, ListItemText, Divider, Box
} from '@mui/material';
import ViewColumnIcon from '@mui/icons-material/ViewColumn';
import DragIndicatorIcon from '@mui/icons-material/DragIndicator';
import BatteryFullIcon from '@mui/icons-material/BatteryFull';
import BatteryStdIcon from '@mui/icons-material/BatteryStd';
import BatteryAlertIcon from '@mui/icons-material/BatteryAlert';
import BatteryChargingFullIcon from '@mui/icons-material/BatteryChargingFull';
import PowerIcon from '@mui/icons-material/Power';
import BoltIcon from '@mui/icons-material/Bolt';

import { useSelector, useDispatch } from 'react-redux';
import { devicesActions } from '../store';

import {
  DndContext,
  closestCenter,
  PointerSensor,
  useSensor,
  useSensors
} from '@dnd-kit/core';
import {
  SortableContext,
  horizontalListSortingStrategy,
  useSortable,
  arrayMove
} from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';

// === funções utilitárias ===
const ONLINE_MIN = 10;
function getStatus(device, position) {
  if (device?.status === 'online' || device?.status === 'offline' || device?.status === 'unknown') {
    return device.status;
  }
  const ts = position?.deviceTime || position?.fixTime || device?.lastUpdate;
  if (!ts) return 'unknown';
  const diffMin = (Date.now() - new Date(ts).getTime()) / 60000;
  return diffMin <= ONLINE_MIN ? 'online' : 'offline';
}
const statusColor = (s) => (s === 'online' ? '#2e7d32' : s === 'offline' ? '#c62828' : '#6b6b6b');
const statusLabel = (s) => (s === 'online' ? 'Conectado' : s === 'offline' ? 'Desconectado' : 'Sem dados');
const toKmh = (s) => (typeof s === 'number' ? (s * 1.852).toFixed(1) : '0.0');
const voltsToPercent = (v) => {
  if (typeof v !== 'number') return null;
  const min = 3.60, max = 4.20;
  const pct = Math.round(((v - min) / (max - min)) * 100);
  return Math.max(0, Math.min(100, pct));
};
const DIR = ['N', 'NNE', 'NE', 'ENE', 'L', 'ESE', 'SE', 'SSE', 'S', 'SSO', 'SO', 'OSO', 'O', 'ONO', 'NO', 'NNO', 'N'];
const headingLabel = (deg) => {
  if (typeof deg !== 'number') return '-';
  const i = Math.round((deg % 360) / 22.5);
  return `${DIR[i]} (${Math.round(deg)}°)`;
};

// === colunas padrão ===
const DEFAULT_COLUMNS = [
  {
    id: 'name',
    label: 'Nome',
    width: 220,
    render: (device, position) => {
      const st = getStatus(device, position);
      return (
        <div style={{ display: 'flex', flexDirection: 'column' }}>
          <Tooltip title={device.name}>
            <span style={{
              fontWeight: 600,
              display: 'block',
              maxWidth: 220,
              whiteSpace: 'nowrap',
              overflow: 'hidden',
              textOverflow: 'ellipsis',
            }}>
              {device.name}
            </span>
          </Tooltip>
          <span style={{ fontSize: '0.6rem', color: statusColor(st) }}>{statusLabel(st)}</span>
        </div>
      );
    },
  },
  {
    id: 'gps',
    label: 'GPS',
    width: 150,
    render: (device, position, dtFmt) =>
      (position?.deviceTime ? dtFmt.format(new Date(position.deviceTime)) : 'Sem posição'),
  },
  { id: 'lat', label: 'Latitude', width: 110, render: (d, p) => (typeof p?.latitude === 'number' ? p.latitude.toFixed(6) : '-') },
  { id: 'lon', label: 'Longitude', width: 110, render: (d, p) => (typeof p?.longitude === 'number' ? p.longitude.toFixed(6) : '-') },
  {
    id: 'speed',
    label: 'Velocidade',
    width: 110,
    align: 'right',
    render: (d, p) => (p ? `${toKmh(p.speed)} km/h` : '-'),
  },
  {
    id: 'address',
    label: 'Endereço',
    width: 420,
    render: (d, p) => p?.address || 'N/D',
  },
  {
    id: 'battery',
    label: 'Bateria',
    width: 110,
    render: (d, p) => {
      const a = p?.attributes || {};
      const pct = typeof a.batteryLevel === 'number' ? a.batteryLevel : voltsToPercent(a.battery);
      const hasBattery = typeof pct === 'number' && !Number.isNaN(pct);
      if (!hasBattery) return '-';
      return (
        <Tooltip title={`Bateria: ${pct}%${typeof a.battery === 'number' ? ` (${a.battery.toFixed(3)} V)` : ''}`}>
          <span style={{ display: 'flex', alignItems: 'center', gap: 4 }}>
            {a.charge ? (
              <BatteryChargingFullIcon style={{ color: 'green', fontSize: 16 }} />
            ) : pct >= 80 ? (
              <BatteryFullIcon style={{ color: 'green', fontSize: 16 }} />
            ) : pct >= 30 ? (
              <BatteryStdIcon style={{ color: 'orange', fontSize: 16 }} />
            ) : (
              <BatteryAlertIcon style={{ color: 'red', fontSize: 16 }} />
            )}
            {`${pct}%`}
          </span>
        </Tooltip>
      );
    },
  },
  {
    id: 'ign',
    label: 'Ign.',
    width: 90,
    render: (d, p) => {
      const ig = p?.attributes?.ignition === true;
      return (
        <Tooltip title={`Ignição: ${ig ? 'Sim' : 'Não'}`}>
          <PowerIcon fontSize="small" color={ig ? 'success' : 'error'} />
        </Tooltip>
      );
    },
  },
  {
    id: 'charge',
    label: 'Carreg.',
    width: 90,
    render: (d, p) => {
      const ch = p?.attributes?.charge === true;
      return (
        <BoltIcon
          fontSize="small"
          style={{ color: ch ? 'green' : 'red' }}
          titleAccess={`Carregando: ${ch ? 'Sim' : 'Não'}`}
        />
      );
    },
  },
];

// === célula de cabeçalho arrastável ===
function SortableHeaderCell({ col }) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({ id: col.id });
  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    cursor: 'grab',
    opacity: isDragging ? 0.6 : 1,
  };
  return (
    <TableCell
      ref={setNodeRef}
      sx={{
        color: '#fff',
        fontWeight: 'bold',
        fontSize: '0.75rem',
        padding: '4px 8px',
        lineHeight: 1.2,
        whiteSpace: 'nowrap',
        backgroundColor: '#007bff',
        userSelect: 'none',
      }}
      align={col.align}
      style={style}
      {...attributes}
      {...listeners}
    >
      <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
        <DragIndicatorIcon fontSize="small" />
        {col.label}
      </Box>
    </TableCell>
  );
}

// === componente principal ===
export default function DeviceTableDraggable({ devices }) {
  const dispatch = useDispatch();
  const positions = useSelector((s) => s.session.positions);

  // 🔥 estilos fixos das células
  const cellStyle = { color: '#000', fontSize: '0.65rem', padding: '4px', whiteSpace: 'nowrap', lineHeight: 1.2 };
  const numCell = { ...cellStyle, textAlign: 'right' };
  const addrCell = { ...cellStyle, whiteSpace: 'normal', maxWidth: 420 };

  const dtFmt = useMemo(
    () => new Intl.DateTimeFormat('pt-BR', { dateStyle: 'short', timeStyle: 'short' }),
    []
  );

  // gerenciamento de colunas
  const STORAGE_KEY = 'deviceTable.columns';
  const [columns, setColumns] = useState(() => {
    const saved = localStorage.getItem(STORAGE_KEY);
    if (saved) {
      try {
        const parsed = JSON.parse(saved);
        const byId = Object.fromEntries(DEFAULT_COLUMNS.map(c => [c.id, c]));
        const order = parsed.order?.filter(id => byId[id]) || DEFAULT_COLUMNS.map(c => c.id);
        const hidden = new Set(parsed.hidden || []);
        return order.map(id => ({ ...byId[id], hidden: hidden.has(id) }));
      } catch { }
    }
    return DEFAULT_COLUMNS.map(c => ({ ...c, hidden: false }));
  });

  useEffect(() => {
    const order = columns.map(c => c.id);
    const hidden = columns.filter(c => c.hidden).map(c => c.id);
    localStorage.setItem(STORAGE_KEY, JSON.stringify({ order, hidden }));
  }, [columns]);

  const visibleCols = columns.filter(c => !c.hidden);
  const colIds = visibleCols.map(c => c.id);

  const [anchorEl, setAnchorEl] = useState(null);
  const openVisMenu = (e) => setAnchorEl(e.currentTarget);
  const closeVisMenu = () => setAnchorEl(null);
  const toggleVisibility = (id) =>
    setColumns(cols => cols.map(c => c.id === id ? { ...c, hidden: !c.hidden } : c));

  // drag and drop
  const sensors = useSensors(useSensor(PointerSensor, { activationConstraint: { distance: 5 } }));
  const onDragEnd = (event) => {
    const { active, over } = event;
    if (!over || active.id === over.id) return;
    setColumns(cols => {
      const onlyVisible = cols.filter(c => !c.hidden);
      const fromIndex = onlyVisible.findIndex(c => c.id === active.id);
      const toIndex = onlyVisible.findIndex(c => c.id === over.id);
      const newVisible = arrayMove(onlyVisible, fromIndex, toIndex);
      const hidden = cols.filter(c => c.hidden);
      return [...newVisible, ...hidden];
    });
  };

  const handleClickRow = (id) => dispatch(devicesActions.selectId(id));

  return (
    <Paper sx={{ borderTop: '2px solid #007bff', backgroundColor: '#fff' }}>
      {/* Cabeçalho fixo */}
      <Table size="small" sx={{ tableLayout: 'fixed', minWidth: 2000, borderCollapse: 'collapse' }}>
        <TableHead>
          <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEnd}>
            <SortableContext items={colIds} strategy={horizontalListSortingStrategy}>
              <TableRow>
                {visibleCols.map(col => (
                  <SortableHeaderCell key={col.id} col={col} />
                ))}
              </TableRow>
            </SortableContext>
          </DndContext>
        </TableHead>
      </Table>

      {/* Corpo rolável */}
      <TableContainer sx={{ maxHeight: '40vh', overflowY: 'auto', overflowX: 'auto' }}>
        <Table size="small" sx={{ tableLayout: 'fixed', minWidth: 2000, borderCollapse: 'collapse' }}>
          <TableBody>
            {devices.map((device) => {
              const position = positions[device.id];
              return (
                <TableRow
                  key={device.id}
                  hover
                  onClick={() => handleClickRow(device.id)}
                  sx={{ cursor: 'pointer', '&:hover': { backgroundColor: '#f2f2f2' } }}
                >
                  {visibleCols.map(col => {
                    const styleBase = col.id === 'address' ? addrCell : (col.align === 'right' ? numCell : cellStyle);
                    return (
                      <TableCell
                        key={col.id}
                        sx={{ ...styleBase, width: col.width, maxWidth: col.width }}
                        align={col.align}
                      >
                        {col.render(device, position, dtFmt)}
                      </TableCell>
                    );
                  })}
                </TableRow>
              );
            })}
          </TableBody>
        </Table>
      </TableContainer>
    </Paper>
  );
}
