import React, { useMemo, useState } from 'react';
import {
  Box, Paper, TextField, FormControl, InputLabel, Select, MenuItem, Button, Stack,
} from '@mui/material';
import { useSelector } from 'react-redux';
import DeviceTableDraggable from './DeviceTableDraggable';

const ONLINE_MIN = 10;

function getStatus(device, position) {
  if (device?.status === 'online' || device?.status === 'offline' || device?.status === 'unknown') {
    return device.status;
  }
  const ts = position?.deviceTime || position?.fixTime || device?.lastUpdate;
  if (!ts) return 'unknown';
  const diffMin = (Date.now() - new Date(ts).getTime()) / 60000;
  return diffMin <= ONLINE_MIN ? 'online' : 'offline';
}

export default function DeviceTableWithFilters({ devices = [] }) {
  const positions = useSelector((state) => state.session.positions);

  const [q, setQ] = useState('');
  const [status, setStatus] = useState('any');
  const [ign, setIgn] = useState('any');
  const [chg, setChg] = useState('any');

  const clearFilters = () => {
    setQ('');
    setStatus('any');
    setIgn('any');
    setChg('any');
  };

  const filtered = useMemo(() => {
    const qNorm = q.trim().toLowerCase();

    return devices.filter((d) => {
      const pos = positions[d.id];
      const attrs = pos?.attributes || {};

      // filtro texto
      if (qNorm) {
        const nameOk = (d.name || '').toLowerCase().includes(qNorm);
        const addrOk = (pos?.address || '').toLowerCase().includes(qNorm);
        if (!nameOk && !addrOk) return false;
      }
      // filtro status
      if (status !== 'any') {
        const st = getStatus(d, pos);
        if (st !== status) return false;
      }
      // filtro ignição
      if (ign !== 'any') {
        const igVal = attrs.ignition === true ? 'on' : attrs.ignition === false ? 'off' : 'any';
        if (igVal !== 'any' && igVal !== ign) return false;
      }
      // filtro carregando
      if (chg !== 'any') {
        const chVal = attrs.charge === true ? 'yes' : attrs.charge === false ? 'no' : 'any';
        if (chVal !== 'any' && chVal !== chg) return false;
      }
      return true;
    });
  }, [devices, positions, q, status, ign, chg]);

  return (
    <Box sx={{ width: '100%' }}>
      {/* === filtros === */}
      <Paper
        elevation={0}
        sx={{
          mb: 0,
          p: 1,
          borderRadius: 1,
          backgroundColor: '#eef6ff',
          border: '1px solid #cfe3ff',
        }}
      >
        <Stack
          direction={{ xs: 'column', md: 'row' }}
          spacing={1}
          alignItems={{ xs: 'stretch', md: 'center' }}
        >
          <TextField
            size="small"
            label="Buscar (nome ou endereço)"
            placeholder="Ex.: João, Av. Paulista…"
            value={q}
            onChange={(e) => setQ(e.target.value)}
            sx={{ minWidth: 260, flex: 2, backgroundColor: '#fff' }}
          />

          <FormControl size="small" sx={{ minWidth: 160, backgroundColor: '#fff' }}>
            <InputLabel>Status</InputLabel>
            <Select label="Status" value={status} onChange={(e) => setStatus(e.target.value)}>
              <MenuItem value="any">Qualquer</MenuItem>
              <MenuItem value="online">Online</MenuItem>
              <MenuItem value="offline">Offline</MenuItem>
              <MenuItem value="unknown">Sem dados</MenuItem>
            </Select>
          </FormControl>

          <FormControl size="small" sx={{ minWidth: 160, backgroundColor: '#fff' }}>
            <InputLabel>Ignição</InputLabel>
            <Select label="Ignição" value={ign} onChange={(e) => setIgn(e.target.value)}>
              <MenuItem value="any">Qualquer</MenuItem>
              <MenuItem value="on">Ligada</MenuItem>
              <MenuItem value="off">Desligada</MenuItem>
            </Select>
          </FormControl>

          <FormControl size="small" sx={{ minWidth: 160, backgroundColor: '#fff' }}>
            <InputLabel>Carregando</InputLabel>
            <Select label="Carregando" value={chg} onChange={(e) => setChg(e.target.value)}>
              <MenuItem value="any">Qualquer</MenuItem>
              <MenuItem value="yes">Sim</MenuItem>
              <MenuItem value="no">Não</MenuItem>
            </Select>
          </FormControl>

          <Box sx={{ flex: 1 }} />
          <Button
            onClick={clearFilters}
            size="small"
            variant="outlined"
            sx={{ whiteSpace: 'nowrap' }}
          >
            Limpar filtros
          </Button>
        </Stack>
      </Paper>

      {/* === tabela com colgroup (DeviceTableDraggable já cuida disso) === */}
      <DeviceTableDraggable devices={filtered} />
    </Box>
  );
}
