import { useState, useEffect } from 'react';
import { useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';
import {
  Toolbar, IconButton, OutlinedInput, InputAdornment,
  Badge, Tooltip, Menu, Checkbox, ListItemText, Divider,
  Box, Typography, Button, FormControl, Select, MenuItem,
  Dialog, DialogTitle, DialogContent
} from '@mui/material';
import { makeStyles } from 'tss-react/mui';
import DescriptionIcon from '@mui/icons-material/Description';
import SettingsIcon from '@mui/icons-material/Settings';
import ExitToAppIcon from '@mui/icons-material/ExitToApp';
import TuneIcon from '@mui/icons-material/Tune';
import PhotoCameraIcon from '@mui/icons-material/PhotoCamera';
import { useTranslation } from '../common/components/LocalizationProvider';

const useStyles = makeStyles()(() => ({
  toolbar: {
    display: 'flex',
    justifyContent: 'space-between',
    padding: '4px 12px',
    minHeight: '48px !important',
    alignItems: 'center',
    position: 'sticky',
    top: 0,
    zIndex: 10,
    backgroundColor: '#fff',
    borderBottom: '1px solid #ddd',
  },
  left: { display: 'flex', alignItems: 'center', gap: 6 },
  center: { flex: 1, display: 'flex', justifyContent: 'center' },
  right: { display: 'flex', alignItems: 'center', gap: 8 },
}));

const statusOptions = [
  { key: 'online', label: 'Online' },
  { key: 'offline', label: 'Offline' },
  { key: 'unknown', label: 'Desconectado' },
];

const MainToolbar = ({ keyword, setKeyword, filter, setFilter }) => {
  const { classes } = useStyles();
  const navigate = useNavigate();
  const t = useTranslation();

  const groups = useSelector((state) => state.groups.items);
  const user = useSelector((state) => state.session.user);

  const [filterAnchorEl, setFilterAnchorEl] = useState(null);

  // Upload logo
  const [openUpload, setOpenUpload] = useState(false);
  const [uploading, setUploading] = useState(false);
  const [selectedFile, setSelectedFile] = useState(null);

  // Logo dinâmico
  const [logoUrl, setLogoUrl] = useState("/logos/default_logo.png");

  useEffect(() => {
    if (user?.id) {
      fetch(`http://rdgsserver.vps-kinghost.net/get_logo.php?userId=${user.id}`)
        .then(res => res.json())
        .then(data => {
          if (data.success && data.logo) {
            setLogoUrl(data.logo);
          }
        })
        .catch(err => console.error("Erro ao carregar logo:", err));
    }
  }, [user]);

  const toggleStatus = (key) => {
    setFilter((prev) => {
      const exists = prev.statuses.includes(key);
      return {
        ...prev,
        statuses: exists ? prev.statuses.filter((s) => s !== key) : [...prev.statuses, key],
      };
    });
  };

  const handleGroupChange = (event) => {
    const groupId = event.target.value;
    setFilter((prev) => ({ ...prev, groups: groupId ? [groupId] : [] }));
  };

  const clearAll = () => setFilter({ statuses: [], groups: [] });

  const handleLogoUpload = async (file) => {
    if (!file || !user?.id) {
      alert("Usuário inválido ou arquivo não selecionado!");
      return;
    }

    setUploading(true);

    const formData = new FormData();
    formData.append('logo', file);
    formData.append('userId', user.id);

    try {
      // chama direto no nginx/php (porta 80)
      const url = `http://rdgsserver.vps-kinghost.net/upload_logo.php`;

      const res = await fetch(url, {
        method: 'POST',
        body: formData,
      });

      const data = await res.json();
      if (data.success) {
        alert('✅ Logo atualizada com sucesso!');
        window.location.reload();
      } else {
        alert('❌ Erro ao atualizar logo: ' + data.error);
      }
    } catch (err) {
      alert('Falha na conexão: ' + err.message);
    }

    setUploading(false);
    setOpenUpload(false);
  };

  return (
    <Toolbar className={classes.toolbar} disableGutters>
      {/* ESQUERDA: LOGO */}
      <div className={classes.left}>
        <img src={logoUrl} alt="Logo Cliente" style={{ height: 36 }} />
        <span style={{ fontSize: '0.85rem', color: '#007bff', fontWeight: 'bold' }}>
          AchaEU PLATAFORMA DE RASTREAMENTO
        </span>
      </div>

      {/* CENTRO: PESQUISA + FILTROS */}
      <div className={classes.center}>
        <OutlinedInput
          placeholder={t('sharedSearchDevices')}
          value={keyword}
          onChange={(e) => setKeyword(e.target.value)}
          endAdornment={
            <InputAdornment position="end">
              <IconButton
                size="small"
                edge="end"
                onClick={(e) => setFilterAnchorEl(e.currentTarget)}
                sx={{
                  backgroundColor: '#e0f0ff',
                  borderRadius: '6px',
                  padding: '4px',
                  '&:hover': { backgroundColor: '#cce4ff' },
                }}
              >
                <Badge
                  color="info"
                  variant="dot"
                  invisible={!filter.statuses.length && !filter.groups.length}
                >
                  <TuneIcon fontSize="small" sx={{ color: '#005baa' }} />
                </Badge>
              </IconButton>
            </InputAdornment>
          }
          size="small"
          sx={{
            height: 36,
            width: '300px',
            backgroundColor: '#e3f2fd',
            borderRadius: '6px',
            border: '1px solid #64b5f6',
            '& input': { fontSize: '0.8rem', padding: '6px 8px' },
          }}
        />
      </div>

      {/* DIREITA: AÇÕES */}
      <div className={classes.right}>
        <Tooltip title="Trocar Logo">
          <IconButton onClick={() => setOpenUpload(true)}>
            <PhotoCameraIcon />
          </IconButton>
        </Tooltip>

        <Tooltip title={t('reportTitle')}>
          <IconButton onClick={() => navigate('/reports/combined')}>
            <DescriptionIcon />
          </IconButton>
        </Tooltip>
        <Tooltip title={t('settingsTitle')}>
          <IconButton onClick={() => navigate('/settings/preferences')}>
            <SettingsIcon />
          </IconButton>
        </Tooltip>
        <Tooltip title={t('loginLogout')}>
          <IconButton onClick={() => navigate('/login')}>
            <ExitToAppIcon />
          </IconButton>
        </Tooltip>
      </div>

      {/* MENU DE FILTRO */}
      <Menu
        anchorEl={filterAnchorEl}
        open={Boolean(filterAnchorEl)}
        onClose={() => setFilterAnchorEl(null)}
        anchorOrigin={{ vertical: 'bottom', horizontal: 'left' }}
        transformOrigin={{ vertical: 'top', horizontal: 'left' }}
      >
        <Box sx={{ px: 1.5, pt: 1 }}>
          <Typography variant="caption" sx={{ fontWeight: 600 }}>
            Status
          </Typography>
        </Box>
        {statusOptions.map((opt) => (
          <MenuItem key={opt.key} onClick={() => toggleStatus(opt.key)}>
            <Checkbox size="small" checked={filter.statuses.includes(opt.key)} />
            <ListItemText primary={opt.label} />
          </MenuItem>
        ))}

        <Divider />

        <Box sx={{ px: 1.5, pt: 1 }}>
          <Typography variant="caption" sx={{ fontWeight: 600 }}>
            Grupo
          </Typography>
        </Box>
        <MenuItem disableRipple disableTouchRipple>
          <FormControl size="small" sx={{ minWidth: 200 }}>
            <Select
              value={filter.groups[0] || ''}
              onChange={handleGroupChange}
              displayEmpty
              renderValue={(v) => (v ? (groups[v]?.name || v) : 'Todos')}
            >
              <MenuItem value="">
                <em>Todos</em>
              </MenuItem>
              {Object.values(groups).map((g) => (
                <MenuItem key={g.id} value={g.id}>
                  {g.name}
                </MenuItem>
              ))}
            </Select>
          </FormControl>
        </MenuItem>

        <Box sx={{ display: 'flex', gap: 1, justifyContent: 'flex-end', px: 1.5, py: 1 }}>
          <Button size="small" onClick={clearAll}>Limpar</Button>
          <Button size="small" variant="contained" onClick={() => setFilterAnchorEl(null)}>Aplicar</Button>
        </Box>
      </Menu>

      {/* DIALOG UPLOAD LOGO */}
      <Dialog open={openUpload} onClose={() => setOpenUpload(false)}>
        <DialogTitle>Trocar Logo</DialogTitle>
        <DialogContent>
          <input
            type="file"
            accept="image/*"
            onChange={(e) => setSelectedFile(e.target.files[0])}
            disabled={uploading}
          />
          <Box sx={{ display: 'flex', justifyContent: 'flex-end', mt: 2, gap: 1 }}>
            <Button onClick={() => setOpenUpload(false)}>Cancelar</Button>
            <Button
              variant="contained"
              onClick={() => handleLogoUpload(selectedFile)}
              disabled={uploading || !selectedFile}
            >
              {uploading ? 'Enviando...' : 'Confirmar'}
            </Button>
          </Box>
        </DialogContent>
      </Dialog>
    </Toolbar>
  );
};

export default MainToolbar;
