import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  Box, Paper, Grid, TextField, Select, MenuItem, Button,
  IconButton, Typography, InputLabel, FormControl
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';

// ===== helpers =====
const ENDPOINT = 'http://rdsgserver.vps-kinghost.net/salvar_venda.php';
const onlyDigits = (v = '') => v.replace(/\D/g, '');

const isValidEmail = (v = '') => /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(v);

const isValidIMEI = (v = '') => {
  const d = onlyDigits(v);
  if (d.length !== 15) return false;
  let sum = 0;
  for (let i = 0; i < 14; i++) {
    let n = parseInt(d[i], 10);
    if (i % 2 === 1) { n *= 2; if (n > 9) n -= 9; }
    sum += n;
  }
  const check = (10 - (sum % 10)) % 10;
  return check === parseInt(d[14], 10);
};

const isValidCPF = (cpf = '') => {
  const s = onlyDigits(cpf);
  if (s.length !== 11 || /^(\d)\1+$/.test(s)) return false;
  let sum = 0;
  for (let i = 0; i < 9; i++) sum += parseInt(s[i], 10) * (10 - i);
  let d1 = (sum * 10) % 11; if (d1 === 10) d1 = 0;
  if (d1 !== parseInt(s[9], 10)) return false;
  sum = 0;
  for (let i = 0; i < 10; i++) sum += parseInt(s[i], 10) * (11 - i);
  let d2 = (sum * 10) % 11; if (d2 === 10) d2 = 0;
  return d2 === parseInt(s[10], 10);
};

// máscaras simples sem libs
const maskCPF = (v) => {
  const d = onlyDigits(v).slice(0, 11);
  return d
    .replace(/^(\d{3})(\d)/, '$1.$2')
    .replace(/^(\d{3})\.(\d{3})(\d)/, '$1.$2.$3')
    .replace(/^(\d{3})\.(\d{3})\.(\d{3})(\d{1,2})$/, '$1.$2.$3-$4');
};
const maskRG = (v) => {
  const d = onlyDigits(v).slice(0, 9);
  return d
    .replace(/^(\d{2})(\d)/, '$1.$2')
    .replace(/^(\d{2})\.(\d{3})(\d)/, '$1.$2.$3')
    .replace(/^(\d{2})\.(\d{3})\.(\d{3})(\d{1})$/, '$1.$2.$3-$4');
};
const maskPhone = (v) => {
  const d = onlyDigits(v).slice(0, 11);
  if (d.length <= 10) {
    return d
      .replace(/^(\d{0,2})(\d{0,4})(\d{0,4}).*/, '($1) $2-$3')
      .trim();
  }
  return d
    .replace(/^(\d{0,2})(\d{0,5})(\d{0,4}).*/, '($1) $2-$3')
    .trim();
};

export default function VendaForm() {
  const navigate = useNavigate();

  const [form, setForm] = useState({
    plano: 'Básico',
    vendedor_id: 1,
    nome: '',
    email: '',
    senha: '',
    rg: '',
    cpf: '',
    telefone: '',
    imei: '',
    dispositivo: '',
    chassi: '',
    renavam: '',
    ano: '',
    valor: 59.9,
  });
  const [errors, setErrors] = useState({});

  const handleChange = (e) => {
    const { name, value } = e.target;

    if (name === 'plano') {
      let valor = 59.9;
      if (value === 'Intermediário') valor = 79.9;
      if (value === 'Premium') valor = 99.9;
      setForm((p) => ({ ...p, plano: value, valor }));
      return;
    }

    // aplica máscaras
    if (name === 'cpf') return setForm((p) => ({ ...p, cpf: maskCPF(value) }));
    if (name === 'telefone') return setForm((p) => ({ ...p, telefone: maskPhone(value) }));
    if (name === 'rg') return setForm((p) => ({ ...p, rg: maskRG(value) }));
    if (name === 'imei') {
      const v = onlyDigits(value).slice(0, 15);
      return setForm((p) => ({ ...p, imei: v }));
    }

    setForm((p) => ({ ...p, [name]: value }));
  };

  const validate = () => {
    const e = {};
    if (!form.nome.trim()) e.nome = 'Obrigatório';
    if (!isValidEmail(form.email)) e.email = 'E-mail inválido';
    if (!form.senha || form.senha.length < 6) e.senha = 'Mínimo 6 caracteres';
    if (!isValidCPF(form.cpf)) e.cpf = 'CPF inválido';
    if (onlyDigits(form.telefone).length < 10) e.telefone = 'Telefone inválido';
    if (!isValidIMEI(form.imei)) e.imei = 'IMEI inválido (15 dígitos)';
    if (!form.dispositivo.trim()) e.dispositivo = 'Obrigatório';
    if (!form.ano || `${form.ano}`.length < 4) e.ano = 'Ano inválido';
    setErrors(e);
    return Object.keys(e).length === 0;
  };

  const salvarVenda = async () => {
    if (!validate()) return;

    const payload = {
      ...form,
      cpf: onlyDigits(form.cpf),
      telefone: onlyDigits(form.telefone),
      imei: onlyDigits(form.imei),
    };

    try {
      const resp = await fetch(ENDPOINT, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        mode: 'cors',
        body: JSON.stringify(payload),
      });
      if (!resp.ok) throw new Error(`HTTP ${resp.status}`);
      const data = await resp.json();
      alert(data.success ? '✅ Venda cadastrada com sucesso!' : `❌ Erro: ${data.error || 'Falha ao cadastrar venda.'}`);
    } catch (e) {
      console.error(e);
      alert('❌ Erro ao comunicar com o servidor!');
    }
  };

  return (
    <Box sx={{ p: 2, pb: 0, height: '100%', boxSizing: 'border-box' }}>
      {/* Cabeçalho */}
      <Box sx={{ display: 'flex', alignItems: 'center', mb: 2, gap: 1 }}>
        <IconButton onClick={() => navigate('/')} size="small">
          <ArrowBackIcon />
        </IconButton>
        <Typography variant="h6">Registrar Venda</Typography>
      </Box>

      <Paper sx={{ p: 2 }}>
        <Grid container spacing={2}>
          <Grid item xs={12} md={6}>
            <TextField fullWidth label="Nome do Cliente" name="nome"
              value={form.nome} onChange={handleChange}
              error={!!errors.nome} helperText={errors.nome} />
          </Grid>
          <Grid item xs={12} md={3}>
            <TextField fullWidth type="email" label="E-mail" name="email"
              value={form.email} onChange={handleChange}
              error={!!errors.email} helperText={errors.email} />
          </Grid>
          <Grid item xs={12} md={3}>
            <TextField fullWidth type="password" label="Senha" name="senha"
              value={form.senha} onChange={handleChange}
              error={!!errors.senha} helperText={errors.senha} />
          </Grid>

          <Grid item xs={12} md={3}>
            <TextField fullWidth label="RG" name="rg"
              value={form.rg} onChange={handleChange} />
          </Grid>
          <Grid item xs={12} md={3}>
            <TextField fullWidth label="CPF" name="cpf"
              value={form.cpf} onChange={handleChange}
              error={!!errors.cpf} helperText={errors.cpf} />
          </Grid>
          <Grid item xs={12} md={3}>
            <TextField fullWidth label="Telefone" name="telefone"
              value={form.telefone} onChange={handleChange}
              error={!!errors.telefone} helperText={errors.telefone} />
          </Grid>
          <Grid item xs={12} md={3}>
            <TextField fullWidth label="IMEI do Dispositivo" name="imei"
              value={form.imei} onChange={handleChange}
              inputProps={{ inputMode: 'numeric', pattern: '[0-9]*', maxLength: 15 }}
              error={!!errors.imei} helperText={errors.imei} />
          </Grid>

          <Grid item xs={12} md={4}>
            <TextField fullWidth label="Nome do Dispositivo" name="dispositivo"
              value={form.dispositivo} onChange={handleChange}
              error={!!errors.dispositivo} helperText={errors.dispositivo} />
          </Grid>
          <Grid item xs={12} md={4}>
            <TextField fullWidth label="Chassi do Veículo" name="chassi" value={form.chassi} onChange={handleChange} />
          </Grid>
          <Grid item xs={12} md={4}>
            <TextField fullWidth label="Renavam do Veículo" name="renavam" value={form.renavam} onChange={handleChange} />
          </Grid>

          <Grid item xs={12} md={2}>
            <TextField fullWidth type="number" label="Ano do Veículo" name="ano"
              value={form.ano} onChange={handleChange}
              error={!!errors.ano} helperText={errors.ano} />
          </Grid>
          <Grid item xs={12} md={3}>
            <FormControl fullWidth>
              <InputLabel id="plano-label">Plano</InputLabel>
              <Select labelId="plano-label" label="Plano" name="plano" value={form.plano} onChange={handleChange}>
                <MenuItem value="Básico">Básico</MenuItem>
                <MenuItem value="Intermediário">Intermediário</MenuItem>
                <MenuItem value="Premium">Premium</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid item xs={12} md={2}>
            <TextField fullWidth type="number" label="Valor do Plano" name="valor"
              value={form.valor} inputProps={{ step: '0.01' }} />
          </Grid>

          <Grid item xs={12} md={5} sx={{ display: 'flex', gap: 1, justifyContent: { xs: 'stretch', md: 'flex-end' } }}>
            <Button variant="contained" onClick={salvarVenda}>Fazer Venda</Button>
            <Button variant="outlined" onClick={() => navigate('/')}>Voltar</Button>
          </Grid>
        </Grid>
      </Paper>
    </Box>
  );
}
